// PWM.v
`timescale 1ps/1fs

// PWM Module
module PWM(
    output reg pwm_out,			// Output to driver for high-side FET
    input master_clock,			// Master clock
    input wire [10:0] count,		// Equivalent to a sawtooth waveform
    input wire signed [19:0] from_pid );
					// Output from PID compensator, which is the
					//   input here for the PWM.  It is actually
					//   a signed value stored in two's
					//   complement format.

    parameter
	delay = 20;

    integer
	max_duty_count = 11'h780;	// This is equivalent to the 4 LSBs in 
					//   an unsigned 11-bit integer to be
					//   all equal to 1's.  This is
					//   equivalen to 1920 in decimal.
					//   This value is used to limit the
					//   max. duty to 1920./2048 ~= 0.9375.

    integer
	diff,
	change;

    initial begin
	pwm_out = 0;
//	$monitor( "Time: %8d, count=%5d, from_pid=%d, diff=%d, pwm_out=%d", $time, count, from_pid, diff, pwm_out );
    end


    always @( posedge master_clock ) begin
	diff = count;
	diff = diff - from_pid;
	change = 0;

	if ( pwm_out ) begin
	    // The output is logic 1 now.  If the input conditions are right,
	    //   need to turrn the output to logic 0 to termiante the PWM
	    //   output pulse.
	    //
	    // The comparison of diff > 0 should have been carried out with
	    //   diff >= 0.  But to be compatible with the existing DPWM
	    //   example employing adv. digital components, diff > 0 is used
	    //   instead.  This means the output pulse has a duration of
	    //   (from_pid + 1) * (clock period of master_clock).
	    if ( diff > 0 || count >= max_duty_count )
		change = 1;
	end
	else begin
	    // The output is logic 0 now.  If count is 0, turn output to
	    //   logic 1 to start the PWM output pulse.
	    if ( count == 0 && diff < 0 )
		change = 1;
	end

	if ( change ) begin
	    pwm_out <= #(delay) ~pwm_out;
	end
    end

endmodule
